/*
 * Broadcom Host Remote Download Utility
 *
 * Linux USB OSL routines.
 *
 * Copyright (C) 2011, Broadcom Corporation
 * All Rights Reserved.
 * 
 * This is UNPUBLISHED PROPRIETARY SOURCE CODE of Broadcom Corporation;
 * the contents of this file may not be disclosed to third parties, copied
 * or duplicated in any form, in whole or in part, without the prior
 * written permission of Broadcom Corporation.
 *
 * $Id: usb_linux.c,v 1.7.30.2 2010-12-15 21:22:42 $
 */

#include <stdio.h>
#include <string.h>
#include <usb.h>
#include <errno.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <typedefs.h>
#include <usbrdl.h>
#include <usbstd.h>
#include "usb_osl.h"

struct usbinfo {
	struct bcm_device_id *devtable;
	int bulkoutpipe;
	int ctrlinpipe;
	usb_dev_handle *dev;
	int bulkinpipe;
};

#define GET_UPPER_16(u32val) ((uint16)((0xFFFF0000 & u32val) >> 16))
#define GET_LOWER_16(u32val) ((uint16)(0x0000FFFF & u32val))

static struct usb_device *
find_my_device(usbinfo_t *info, struct bcm_device_id **bd)
{
	struct usb_bus *bus;
	struct usb_device *device;
	struct bcm_device_id *bcmdev;

	/* Enumerate all busses */
	usb_find_busses();

	/* Enumerate all devices */
	usb_find_devices();

	/* Find the right Vendor/Product pair */
	for (bus = usb_busses; bus; bus = bus->next)
		for (device = bus->devices; device; device = device->next) {
			printf("Vendor 0x%x ID 0x%x\n", device->descriptor.idVendor,
			       device->descriptor.idProduct);
			for (bcmdev = info->devtable; bcmdev->name; bcmdev++)
				if ((device->descriptor.idVendor == bcmdev->vend) &&
				    (device->descriptor.idProduct == bcmdev->prod)) {
					*bd = bcmdev;
					return device;
				}
		}
	return NULL;
}

usbinfo_t *
usbdev_init(struct bcm_device_id *devtable, struct bcm_device_id **bcmdev)
{
	struct usb_device *device;
	int status;
	usb_dev_handle *dev;
	struct usb_config_descriptor *config;
	struct usb_interface *interface;
	struct usb_interface_descriptor *altsetting;
	struct usb_endpoint_descriptor *endpoint;
	int i;
	usbinfo_t *info;

	if ((info = malloc(sizeof(usbinfo_t))) == NULL)
		return NULL;
	bzero(info, sizeof(usbinfo_t));

	info->devtable = devtable;

	/* Initialize USB subsystem */
	usb_init();

	device = find_my_device(info, bcmdev);
	if (device == NULL) {
		*bcmdev = NULL;
		printf("No devices found\n");
		goto err;
	}

	/* Open the USB device */
	if (!(dev = usb_open(device))) {
		printf("Failed to open device\n");
		goto err;
	}

	info->dev = dev;

	if ((status = usb_set_configuration(dev, 1))) {
		printf("Failed to set configuartion\n");
		goto err;
	}

	/*
	 * Try to claim the interface; disregard status since this
	 * call will fail on some versions of Linux
	 */
	usb_claim_interface(dev, 0);

	config = device->config;
	if (!config)
		goto err;

	/* last interface is the one with the bulk out */
	interface = &config->interface[config->bNumInterfaces - 1];
	altsetting = interface->altsetting;
	if (!altsetting)
		goto err;

	endpoint = altsetting->endpoint;
	if (!endpoint)
		goto err;

	/* Check data endpoints */
	for (i = 0; i < altsetting->bNumEndpoints; i++) {
		if (UE_GET_XFERTYPE(endpoint[i].bmAttributes) != UE_BULK)
			continue;
		if (((UE_GET_DIR(endpoint[i].bEndpointAddress) == UE_DIR_IN))) {
			if (!info->bulkinpipe)
				info->bulkinpipe = UE_GET_ADDR(endpoint[i].bEndpointAddress);
		}
		if (!((UE_GET_DIR(endpoint[i].bEndpointAddress) == UE_DIR_IN))) {
			info->bulkoutpipe = UE_GET_ADDR(endpoint[i].bEndpointAddress);
		}
	}

	info->ctrlinpipe = USB_CTRL_IN;

	if (info->bulkoutpipe == 0) {
		printf("bcmdl: could not find bulk out ep on interface %d\n",
		       config->bNumInterfaces - 1);
		goto err;
	}

	if (info->bulkinpipe == 0) {
		printf("bcmdl: could not find bulk in ep on interface %d\n",
		       config->bNumInterfaces - 1);
		goto err;
	}

	return info;
err:
	free(info);
	return NULL;
}

int
usbdev_deinit(usbinfo_t *info)
{
	int status;

	/* Release the interface */
	if ((status = usb_release_interface(info->dev, 0)))
		printf("Failed to release device %d\n", status);

	/* Close the USB device */
	if (usb_close(info->dev))
		printf("failed to close device\n");

	free(info);
	return 0;
}

int
usbdev_bulk_write(usbinfo_t *info, void *data, int len, int timeout)
{
	return usb_bulk_write(info->dev, info->bulkoutpipe, (void *)data, len, timeout);

}

int
usbdev_bulk_read(usbinfo_t *info, void *data, int len, int timeout)
{
	return usb_bulk_read(info->dev, info->bulkinpipe, (void *)data, len, timeout);
}

int
usbdev_control_read(usbinfo_t *info, int request, int value, int index,
                    void *data, int size, bool interface, int timeout)

{
	if (interface)
		return usb_control_msg(info->dev, info->ctrlinpipe, request, value, index,
		                       (char*)data, size, timeout);
	else
		return usb_control_msg(info->dev, 0xc0, request, value, index,
		                       (char*)data, size, timeout);
}

/* special fun to talk to 4319 cpuless usb device */
int
usbdev_control_write(usbinfo_t *info, int request, int value, int index,
                     void *data, int size, bool interface, int timeout)

{
	if (interface)
		return usb_control_msg(info->dev, 0x41, request, value, index,
		                       (char*)data, size, timeout);
	else
		return usb_control_msg(info->dev, 0x40, request, value, index,
		                       (char*)data, size, timeout);
}

/* To reset the USB device. Used for HSIC hot plugging */
int usbdev_reset(usbinfo_t *info)
{
	return usb_reset(info->dev);
}
